const database = require('./database');
const config = require('../config/config');
const translator = require('./translator');

class NotificationManager {
  constructor() {
    this.client = null;
    this.ticketTimers = new Map();
    this.agentTimers = new Map();
    this.adminNumbers = config.admins;
  }

  /**
   * Set WPPConnect client
   */
  setClient(client) {
    this.client = client;
  }

  /**
   * Start ticket monitoring
   */
  startTicketMonitoring(ticketId, clientPhone) {
    // Clear previous timer if exists
    this.stopTicketMonitoring(ticketId);

    // Set timeout to notify administrators
    const timeoutMinutes = config.notifications.ticketTimeout;
    const timerId = setTimeout(async () => {
      await this.notifyAdminsTicketTimeout(ticketId, clientPhone);
    }, timeoutMinutes * 60 * 1000);

    this.ticketTimers.set(ticketId, timerId);
    console.log(`⏰ Monitoring started for ticket #${ticketId} (timeout: ${timeoutMinutes} min)`);
  }

  /**
   * Stop ticket monitoring
   */
  stopTicketMonitoring(ticketId) {
    const timerId = this.ticketTimers.get(ticketId);
    if (timerId) {
      clearTimeout(timerId);
      this.ticketTimers.delete(ticketId);
      console.log(`✅ Monitoring stopped for ticket #${ticketId}`);
    }
  }

  /**
   * Start agent response monitoring
   */
  startAgentResponseMonitoring(ticketId, agentPhone) {
    // Clear previous timer if exists
    this.stopAgentResponseMonitoring(ticketId);

    // Set timeout for agent reminder
    const timeoutMinutes = config.notifications.agentResponseTimeout;
    const timerId = setTimeout(async () => {
      await this.remindAgent(ticketId, agentPhone);
    }, timeoutMinutes * 60 * 1000);

    this.agentTimers.set(ticketId, timerId);
  }

  /**
   * Stop agent response monitoring
   */
  stopAgentResponseMonitoring(ticketId) {
    const timerId = this.agentTimers.get(ticketId);
    if (timerId) {
      clearTimeout(timerId);
      this.agentTimers.delete(ticketId);
    }
  }

  /**
   * Notify administrators about unattended ticket
   */
  async notifyAdminsTicketTimeout(ticketId, clientPhone) {
    const ticket = database.getTicket(ticketId);
    
    if (!ticket || ticket.status !== 'open' || ticket.agent_name) {
      return; // Ticket already handled or closed
    }

    const message = `🚨 *ALERT: Unattended Ticket*

🎫 Ticket: #${ticketId}
👤 Client: ${ticket.client_name}
📞 Phone: ${clientPhone}
⏰ Waiting time: ${config.notifications.ticketTimeout} minutes

⚠️ This ticket has not been accepted by any agent yet!

Use: /atender or /attend ${clientPhone}`;

    // Register notification in database
    for (const adminNumber of config.notifications.adminNumbers) {
      const notificationId = database.createNotification(
        ticketId,
        'ticket_timeout',
        adminNumber,
        message
      );

      // Send notification
      if (this.client) {
        try {
          await this.client.sendText(adminNumber + '@c.us', message);
          database.markNotificationSent(notificationId);
          console.log(`📢 Notification sent to admin ${adminNumber}`);
        } catch (error) {
          console.error(`❌ Error sending notification to ${adminNumber}:`, error.message);
        }
      }
    }
  }

  /**
   * Send reminder to agent
   */
  async remindAgent(ticketId, agentPhone) {
    const ticket = database.getTicket(ticketId);
    
    if (!ticket || ticket.status !== 'open') {
      return; // Ticket was closed
    }

    const message = `⏰ *Service Reminder*

🎫 Ticket: #${ticketId}
👤 Client: ${ticket.client_name}

The client has been waiting for your response for ${config.notifications.agentResponseTimeout} minutes.

💡 Respond as soon as possible for better customer experience.`;

    // Register notification
    const notificationId = database.createNotification(
      ticketId,
      'agent_reminder',
      agentPhone,
      message
    );

    // Send reminder
    if (this.client) {
      try {
        await this.client.sendText(agentPhone + '@c.us', message);
        database.markNotificationSent(notificationId);
        console.log(`📢 Reminder sent to agent ${agentPhone}`);
      } catch (error) {
        console.error(`❌ Error sending reminder to ${agentPhone}:`, error.message);
      }
    }
  }

  /**
   * Notify client about service start
   */
  async notifyClientAgentStarted(clientPhone, agentName, ticketId, language = 'pt') {
    const messageEN = `✅ *Service Started*

Hello! Agent *${agentName}* is now assisting you.

🎫 Ticket: #${ticketId}

Feel free to send your questions! 😊

💡 To close this ticket, use: /close ticket`;

    const translator = require('./translator');
    const message = await translator.translateMessage(messageEN, language);

    if (this.client) {
      try {
        await this.client.sendText(clientPhone + '@c.us', message);
        console.log(`✅ Start notification sent to client ${clientPhone}`);
      } catch (error) {
        console.error(`❌ Error notifying client ${clientPhone}:`, error.message);
      }
    }
  }

  /**
   * Notify client about ticket closure
   */
  async notifyClientTicketClosed(clientPhone, ticketId, language = 'pt', reason = null) {
    let messageEN = `✅ *Ticket Closed*

Your ticket #${ticketId} has been closed.`;

    if (reason) {
      messageEN += `\n\n📝 Reason: ${reason}`;
    }

    messageEN += `\n\nFor more help, open a new ticket with */open ticket*

Thank you for using our services! 🙏

*${config.bot.company}* - Always by your side! 💙`;

    const translator = require('./translator');
    const message = await translator.translateMessage(messageEN, language);

    if (this.client) {
      try {
        await this.client.sendText(clientPhone + '@c.us', message);
        console.log(`✅ Closure notification sent to client ${clientPhone}`);
      } catch (error) {
        console.error(`❌ Error notifying client ${clientPhone}:`, error.message);
      }
    }
  }

  /**
   * Notify agents about ticket closed by client
   */
  async notifyAgentsClientClosedTicket(ticketId, clientPhone, clientName, agentPhone = null, reason = null) {
    const translator = require('./translator');
    
    let messageEN = `🔔 *Client Closed Ticket*

📋 Ticket: #${ticketId}
👤 Client: ${clientName} (${clientPhone})`;

    if (reason) {
      messageEN += `\n📝 Reason: ${reason}`;
    }

    messageEN += `\n\n💡 The client closed the service.`;

    const message = await translator.translateSystemMessage(messageEN);

    // If there's a specific agent handling, notify them
    if (agentPhone && this.client) {
      try {
        await this.client.sendText(agentPhone + '@c.us', message);
        console.log(`✅ Agent ${agentPhone} notified about closure`);
      } catch (error) {
        console.error(`❌ Error notifying agent:`, error.message);
      }
    }

    // Notify all admins
    for (const adminNumber of this.adminNumbers) {
      if (adminNumber !== agentPhone) {
        try {
          await this.client.sendText(adminNumber + '@c.us', message);
        } catch (error) {
          console.error(`❌ Error notifying admin ${adminNumber}:`, error.message);
        }
      }
    }
  }

  /**
   * Notify agent about pending transfer
   */
  async notifyAgentTransferPending(targetAgent, ticketId, fromAgent, clientPhone, clientName, reason) {
    if (!this.client) return;

    const messageEN = `📨 *Incoming Transfer*

📋 Ticket: #${ticketId}
👤 Client: ${clientName}
📞 Phone: ${clientPhone}
👥 From Agent: ${fromAgent}

📝 *Transfer Reason:*
${reason}

💡 Type */accept* to handle this client`;

    const message = await translator.translateSystemMessage(messageEN);

    try {
      await this.client.sendText(targetAgent + '@c.us', message);
      console.log(`✅ Transfer notification sent to agent ${targetAgent}`);
    } catch (error) {
      console.error(`❌ Error notifying agent:`, error.message);
    }
  }

  /**
   * Notify client about transfer
   */
  async notifyClientTransferred(clientPhone, newAgentPhone, language) {
    if (!this.client) return;

    const messageEN = `🔄 *Your conversation is being transferred*

Please wait a moment while another agent takes over your case.

Thank you for your patience! 🙏`;

    const message = await translator.translateMessage(messageEN, language);

    try {
      await this.client.sendText(clientPhone + '@c.us', message);
      console.log(`✅ Client ${clientPhone} notified about transfer`);
    } catch (error) {
      console.error(`❌ Error notifying client:`, error.message);
    }
  }

  /**
   * Notify original agent that transfer was accepted
   */
  async notifyAgentTransferAccepted(fromAgent, newAgent, clientName) {
    if (!this.client) return;

    const messageEN = `✅ *Transfer Accepted*

Agent ${newAgent} has accepted the transfer.

👤 Client: ${clientName}

The conversation has been successfully transferred! 🎉`;

    const message = await translator.translateSystemMessage(messageEN);

    try {
      await this.client.sendText(fromAgent + '@c.us', message);
      console.log(`✅ Agent ${fromAgent} notified about transfer acceptance`);
    } catch (error) {
      console.error(`❌ Error notifying agent:`, error.message);
    }
  }

  /**
   * Notify client about new agent
   */
  async notifyClientNewAgent(clientPhone, newAgentName, language) {
    if (!this.client) return;

    const messageEN = `👋 *New Agent in Conversation*

Hello! I am ${newAgentName}, your new support agent and I'm here to help.

I already have all the details of your case. How can I help you?`;

    const message = await translator.translateMessage(messageEN, language);

    try {
      await this.client.sendText(clientPhone + '@c.us', message);
      console.log(`✅ Client ${clientPhone} notified about new agent: ${newAgentName}`);
    } catch (error) {
      console.error(`❌ Error notifying client:`, error.message);
    }
  }

  /**
   * Send pending notifications
   */
  async sendPendingNotifications() {
    if (!this.client) {
      return;
    }

    const pending = database.getPendingNotifications();
    
    for (const notification of pending) {
      try {
        await this.client.sendText(
          notification.recipient + '@c.us',
          notification.message
        );
        database.markNotificationSent(notification.id);
        console.log(`✅ Pending notification sent: ${notification.notification_type}`);
      } catch (error) {
        console.error(`❌ Error sending pending notification:`, error.message);
      }
    }
  }

  /**
   * Clear all timers
   */
  clearAllTimers() {
    // Clear ticket timers
    for (const [ticketId, timerId] of this.ticketTimers.entries()) {
      clearTimeout(timerId);
    }
    this.ticketTimers.clear();

    // Clear agent timers
    for (const [ticketId, timerId] of this.agentTimers.entries()) {
      clearTimeout(timerId);
    }
    this.agentTimers.clear();

    console.log('✅ All notification timers have been cleared');
  }
}

module.exports = new NotificationManager();
